c     **************************************************************
c     The matrix equation for the number densities of the level
c     populations is solved by LU decomposition with improvement.
c     Here, we use the routines ludcmp, lubksb, and mprove from the
c     'Numerical recipies in FORTRAN'
c     **************************************************************
c     They were slightly modified to make one routine from
c     the three parts and to reduce the number of parameters to
c     those really necessary for our problem
c     **************************************************************
      SUBROUTINE lusolver(a,n,b)

      implicit none
      integer n
      include 'fsizes.inc'
c     **************************************************************
c     declarations for the first part
c     a is the matrix (will be destroyed)
c     n is the really used dimension of the matrix 
c     nlev is the physical dimension of all vectors
c     index is a vector of permutations 
c      - will be used by the second part
c     **************************************************************
      INTEGER i,imax,j,k,indx(nlev)
      REAL*8 a(nlev,nlev),astor(nlev,nlev),TINY
      REAL*8 aamax,dum,sum,vv(nlev)
      PARAMETER (TINY=1.0d-30)

c     **************************************************************
c     declarations for the second and third part
c     b is the vector with inhomogeneous part 
c      - will be returned as the solution vector
c     n is the used dimension of the vector
c     **************************************************************
      REAL*8 b(nlev)
      REAL*8 bstor(nlev),x(nlev),sdp
      INTEGER ii,ll

c     **************************************************************
c     First save a and b for later improvement
c     **************************************************************
      do i=1,n
        do j=1,n
          astor(i,j)=a(i,j)
        enddo
        bstor(i)=b(i)
      enddo

c     **************************************************************
c     First part from Numerical recipies - matrix decomposition
c     according to ludcmp
c     **************************************************************
      do 12 i=1,n
        aamax=0d0
        do 11 j=1,n
          if (abs(a(i,j)).gt.aamax) aamax=abs(a(i,j))
11      continue
        if (aamax.eq.0.) pause 'Singular matrix in LU decomposition'
        vv(i)=1d0/aamax
12    continue
      do 19 j=1,n
        do 14 i=1,j-1
          sum=a(i,j)
          do 13 k=1,i-1
            sum=sum-a(i,k)*a(k,j)
13        continue
          a(i,j)=sum
14      continue
        aamax=0d0
        do 16 i=j,n
          sum=a(i,j)
          do 15 k=1,j-1
            sum=sum-a(i,k)*a(k,j)
15        continue
          a(i,j)=sum
          dum=vv(i)*abs(sum)
          if (dum.ge.aamax) then
            imax=i
            aamax=dum
          endif
16      continue
        if (j.ne.imax)then
          do 17 k=1,n
            dum=a(imax,k)
            a(imax,k)=a(j,k)
            a(j,k)=dum
17        continue
          vv(imax)=vv(j)
        endif
        indx(j)=imax
        if(a(j,j).eq.0d0)a(j,j)=TINY
        if(j.ne.n)then
          dum=1d0/a(j,j)
          do 18 i=j+1,n
            a(i,j)=a(i,j)*dum
18        continue
        endif
19    continue

c     **************************************************************
c     Second part - solution of the two triagular matrices
c     by forward and back substitutaion
c     according to lubksb
c     **************************************************************
      ii=0
      do 22 i=1,n
        ll=indx(i)
        sum=b(ll)
        b(ll)=b(i)
        if (ii.ne.0)then
          do 21 j=ii,i-1
            sum=sum-a(i,j)*b(j)
21        continue
        else if (sum.ne.0.) then
          ii=i
        endif
        b(i)=sum
22    continue
      do 24 i=n,1,-1
        sum=b(i)
        do 23 j=i+1,n
          sum=sum-a(i,j)*b(j)
23      continue
        b(i)=sum/a(i,i)
24    continue

c     **************************************************************
c     Third part - imrovement (1 step only)
c     according to mprove
c     Only executed for large matrices
c     **************************************************************
      if (n.lt.10) return
      do 30 i=1,n
        x(i)=b(i)
30    continue
      do 32 i=1,n
        sdp=-bstor(i)
        do 31 j=1,n
          sdp=sdp+astor(i,j)*x(j)
31      continue
        b(i)=sdp
32    continue

c     **************************************************************
c     Repeat second part (according to lubksb)
c     **************************************************************
      ii=0
      do 42 i=1,n
        ll=indx(i)
        sum=b(ll)
        b(ll)=b(i)
        if (ii.ne.0)then
          do 41 j=ii,i-1
            sum=sum-a(i,j)*b(j)
41        continue
        else if (sum.ne.0.) then
          ii=i
        endif
        b(i)=sum
42    continue
      do 44 i=n,1,-1
        sum=b(i)
        do 43 j=i+1,n
          sum=sum-a(i,j)*b(j)
43      continue
        b(i)=sum/a(i,i)
44    continue

c     *************************************************************
c     Compute improvement
c     *************************************************************
      do 33 i=1,n
c        b(i)=x(i)-b(i)
        b(i)=x(i)
33    continue

      return
      end

