/*
 *    Starlight Xpress Camera Driver For Video4Linux.
 *              (ICX027 CCD)
 *
 *	This version only works as a module.
 *
 *	Video4Linux conversion work by Alan Cox.
 *	Parport compatibility by Phil Blundell.
 *	Busy loop avoidance by Mark Cooke.
 *
 *    Module parameters:
 *
 *	yieldlines=<1 - 250>
 *
 *	  When acquiring a frame from the camera, the data gathering
 *	  loop will yield back to the scheduler after completing
 *	  this many lines. The default of 4 provides a trade-off
 *	  between increased frame acquisition time and impact on
 *	  interactive response.
 */

/******************************************************************

Copyright (C) 1996 by Scott Laird

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL SCOTT LAIRD BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#define MODULE

/* retrieve the CONFIG_* macros */
#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif


#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/parport.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <linux/sched.h>
#include <asm/segment.h>
#include <linux/wrapper.h>
#include <linux/version.h>
#include <linux/videodev.h>
#include <linux/vmalloc.h>
#include <asm/semaphore.h>
#include <asm/uaccess.h>

#include "sxcam.h"

static unsigned int yieldlines=4;  /* Yield after this many during capture */


#if LINUX_VERSION_CODE < 0x020000
char kernel_version[] = UTS_RELEASE;		/* Required by module */
#endif

#if LINUX_VERSION_CODE >= 0x020117
MODULE_PARM(yieldlines,"i");
#endif

struct sxcam_device {
	struct video_device vdev;
	struct pardevice *pdev;
	struct parport *pport;
	int ccd_width, ccd_height;
	int width, height;
	int top, left;
	int depth;
	int palette;
	int exptime;
	unsigned short *buffer;
	struct semaphore lock;
};

#undef LINE

static unsigned short pixarray[SXHEIGHT][SXWIDTH]; /* GFP_BUFFER */
//static unsigned short *pixarray=NULL;

#ifdef LINE
static unsigned short pixbuff[512];
#endif

#define WZ 0
#define pause(Zeit) {long int i; for(i=0; i<Zeit;i++) {}while(0);}

static void sxcam_clearer(struct sxcam_device *);
static void sxcam_vert(struct sxcam_device *);
static void sxcam_writeintoarray(struct sxcam_device *, int, int, unsigned short);
static unsigned short sxcam_readfromarray(struct sxcam_device *, int, int);
static void sxcam_linedump(struct sxcam_device *);
static void sxcam_horclear(struct sxcam_device *);
static void sxcam_reader(struct sxcam_device *);
static void sxcam_expose(struct sxcam_device *, long);
static void sxcam_lineread(struct sxcam_device *);

static ypos=0;

/* read out the next pixel line from adc */

static void sxcam_lineread(struct sxcam_device *sx)
{
	unsigned int x;
	unsigned char lo=0,me=0,hi=0;
	unsigned short d1;

	parport_write_data(sx->pport, 131);
	for(x=0; x<sx->ccd_width; x++) {
		parport_write_data(sx->pport, 3);
		pause(WZ);
		parport_write_data(sx->pport, 131);
		pause(WZ);
		
		/* read only between start and stop */
		if(x>=sx->left && x<=(sx->left+sx->width)) {
			pause(WZ);
			if( sx->depth == 16) {
				parport_write_control(sx->pport, 0x2);
				lo=((unsigned char)parport_read_status(sx->pport) & 120);
			}
			parport_write_control(sx->pport, 0x0);
			me=((unsigned char)parport_read_status(sx->pport) & 120);
			parport_write_control(sx->pport, 0x4);
			hi=((unsigned char)parport_read_status(sx->pport) & 120);
//			parport_write_data(sx->pport, 3);
			pause(WZ);
			d1=(lo*2 + (me*32) + (hi*512)); /* scale to full 0xfff0 */
#ifndef LINE
			sxcam_writeintoarray(sx,x,ypos,d1);
#else
			pixbuff[x-start]=d1*256+d1/256;
#endif
		}
	}
}


static void sxcam_clearer(struct sxcam_device *sx)
{
	parport_write_data(sx->pport, 3);
	parport_write_data(sx->pport, 67);
	pause(WZ);
	parport_write_data(sx->pport, 3);
	pause(WZ);

}

static void sxcam_vert(struct sxcam_device *sx)
{
	parport_write_data(sx->pport, 131);
	parport_write_data(sx->pport, 135);
	pause(WZ);
	parport_write_data(sx->pport, 134);
	pause(WZ);
	parport_write_data(sx->pport, 142);
	pause(WZ);
	parport_write_data(sx->pport, 140);
	pause(WZ);
	parport_write_data(sx->pport, 141);
	pause(WZ);
	parport_write_data(sx->pport, 137);
	pause(WZ);
	parport_write_data(sx->pport, 139);
	pause(WZ);
	parport_write_data(sx->pport, 131);
	pause(WZ);
	parport_write_data(sx->pport, 3);
	pause(WZ);
}

/* store pixel in array and take care for correct endianness */

static void sxcam_writeintoarray(struct sxcam_device *sx, int x, int y, unsigned short val)
{
//	unsigned short d2;
//	d2=val/256+val*256;
	pixarray[y][x]=val;
//	sx->buffer[x+(sx->ccd_width)*y]=val;
}

/* read a pixel */

static unsigned short sxcam_readfromarray(struct sxcam_device *sx, int x, int y)
{
	unsigned short d2;
	d2=pixarray[y][x];
//	d2=sx->buffer[x+(sx->ccd_width)*y];
//	d2=d2/256+d2*256;
	return d2;
}

/* skip the next line as fast as possible */

static void sxcam_linedump(struct sxcam_device *sx)
{
	int y;
	parport_write_data(sx->pport, 131);
	for(y=0;y<538;y++) {
		parport_write_data(sx->pport, 3);
		parport_write_data(sx->pport, 131);
	}
}

/* clear the ccd completely */

static void sxcam_horclear(struct sxcam_device *sx)
{
	int y;
	for(y=0; y<550;y++) {
		pause(WZ);
		parport_write_data(sx->pport, 131);
		pause(WZ);
		parport_write_data(sx->pport, 3);
		pause(WZ);
	}
}

static void sxcam_reader(struct sxcam_device *sx)
{
	parport_write_data(sx->pport, 5);
	pause(WZ);
	parport_write_data(sx->pport, 21);
	pause(WZ);
	parport_write_data(sx->pport, 5);
	pause(WZ);
	parport_write_data(sx->pport, 37);
	pause(WZ);
	parport_write_data(sx->pport, 5);
	pause(WZ);
	parport_write_data(sx->pport, 7);
	pause(WZ);
	parport_write_data(sx->pport, 6);
	pause(WZ);
	parport_write_data(sx->pport, 14);
	pause(WZ);
	parport_write_data(sx->pport, 12);
	pause(WZ);
	parport_write_data(sx->pport, 13);
	pause(WZ);
	parport_write_data(sx->pport, 9);
	pause(WZ);
	parport_write_data(sx->pport, 11);
	pause(WZ);
	parport_write_data(sx->pport, 3);
	pause(WZ);
}

static void long_delay (long tm)
{
//    mdelay(tm); /* delay in msec */
	current->state=TASK_INTERRUPTIBLE;
	schedule_timeout(tm*HZ/1000);
}

/*
static struct {
	unsigned int xstart,ystart;
	unsigned int xstop,ystop;
} sxctl = {23,8,523,298};
*/

static void sxcam_expose(struct sxcam_device *sx, long tm)
{
	int i;

	for(i=0; i<700; i++) /* clear ccd - no idea why 700 makes it */
		sxcam_vert(sx);

	for(i=0;i<9;i++)   /* see above */
		sxcam_horclear(sx);

	parport_write_control(sx->pport, 8); /* Amplifier off */
	sxcam_clearer(sx);
	long_delay(tm);    /* Time to integrate - is busy loop necessary ? */
	sxcam_reader(sx);
	parport_write_control(sx->pport, 0); /* Amplifier on */


	/* This skips the first shaded pixels */
	for(ypos=0;ypos<sx->ccd_height;ypos++)
	{
		if( (ypos%yieldlines)==0 ) {
			current->state=TASK_INTERRUPTIBLE;
			schedule_timeout(HZ/20000);
		}
		sxcam_vert(sx);
		if( ypos > sx->top && ypos < (sx->top+sx->height) )
			sxcam_lineread(sx);
		else
			sxcam_linedump(sx);
	}

#ifndef LINE
	/* Well ... */
//	for(i=0;i<49;i++)
//		sxcam_vert(sx);
#endif
}

/*
static int sxcam_read_data(struct sxcam_device *sx)
{
	return 0;
}

static int sxcam_write_data(struct sxcam_device *sx, unsigned int data)
{
	return 0;
}

static inline int sxcam_set(struct sxcam_device *sx, unsigned int cmd, unsigned int data)
{
	return 0;
}

static inline int sxcam_get(struct sxcam_device *sx, unsigned int cmd)
{
	return 0;
}
*/
static int sxcam_detect(struct sxcam_device *sx)
{
	/* no detection code known */
	return 1;
}

static void sxcam_reset(struct sxcam_device *sx)
{
	/* Clear CCD Buffer */
//	parport_write_control(sx->pport, 0xc);
//	parport_write_control(sx->pport, 0x8);
//	mdelay(1);
//	parport_write_control(sx->pport, 0xc);
//	mdelay(1);          
}
/*
static void sxcam_setup(struct sxcam_device *sx)
{
	sxcam_reset(sx);
}
*/
/* Read some bytes from the camera and put them in the buffer. 
   nbytes should be a multiple of 3, because bidirectional mode gives
   us three bytes at a time.  */
/*
static unsigned int sxcam_read_bytes(struct sxcam_device *sx, unsigned char *buf, unsigned int nbytes)
{
	return 0;
}
*/
#define BUFSZ   150

static long sxcam_capture(struct sxcam_device *sx, unsigned char *buf, unsigned long len)
{
	/* Copy  len bytes of the frame into buf */
	int r;
//	unsigned lines, pixelsperline, bitsperxfer;
//	size_t wantlen, outptr = 0;
//	char tmpbuf[BUFSZ];
//	int filepos=0;
	unsigned long bufsize;
	unsigned short pixel;

	if (verify_area(VERIFY_WRITE, buf, len))
		return -EFAULT;

	sx->buffer = vmalloc((sx->ccd_width)*(sx->ccd_height));
//	printk(KERN_ERR "%x\n",sx->buffer);

	bufsize = (sx->width)*(sx->height);
	if(sx->depth==16) bufsize*=2;
	else if(sx->depth==24) bufsize*=3;

	if(bufsize<len) len=bufsize;

//	printk(KERN_ERR "sxcam: Exposing image %d milliseconds into %d bytes. %d %d %d %d\n",sx->exptime,bufsize,sx->left,sx->top,sx->width,sx->height);
	sxcam_expose(sx,sx->exptime);

	for (r = 0; r < len; r++) {
#ifdef LINE
		if(filepos>=0 && filepos<(2*512*290) ) {
			if( (filepos%(512*2))==0 ) {
				vert();
				lineread(xstart,xstop);
			}

			if(filepos%2)
				put_user((unsigned char)pixbuff[(filepos%(512*2))/2], buf + r);
			else
				put_user((unsigned char)(pixbuff[(filepos%(512*2))/2]/256), buf + r);
			filepos+=1;
		} else {
			put_user(0x00, buf + r);
		}
#else
		if(sx->depth==16) {
			pixel=sxcam_readfromarray(sx,sx->left+(r/2)%(sx->width),sx->top+(r/2)/(sx->width));
			if(r%2)
				put_user((unsigned char)(pixel/256), buf + r);
			else
				put_user((unsigned char)(pixel), buf + r);
		} else if(sx->depth==8) {
			pixel=sxcam_readfromarray(sx,sx->left+(r)%(sx->width),sx->top+(r)/(sx->width));
			put_user((unsigned char)(pixel/256), buf + r);
		} else if(sx->depth==24 && sx->palette==VIDEO_PALETTE_RGB24) {
			pixel=sxcam_readfromarray(sx,sx->left+(r/3)%(sx->width),sx->top+(r/3)/(sx->width));
			if(r%3==0)
				put_user((unsigned char)(pixel/256), buf + r);
			else if(r%3==1)
				put_user((unsigned char)(pixel/256), buf + r);
			else
				put_user((unsigned char)(pixel/256), buf + r);
		}
#endif								
	}

	vfree(sx->buffer);

	return r;

//	return len;
}


static int sxcam_open(struct video_device *dev, int flags)
{
	struct sxcam_device *sx=(struct sxcam_device *)dev;

	down(&sx->lock);
//	sx->buffer = vmalloc((sx->ccd_width)*(sx->ccd_height));
//	if(!(sx->buffer))
//		goto out_mem;

	up(&sx->lock);
	MOD_INC_USE_COUNT;
	return 0;
//out_mem:
//	up(&sx->lock);
//	return -ENOMEM;
}

static void sxcam_close(struct video_device *dev)
{
	struct sxcam_device *sx=(struct sxcam_device *)dev;
	down(&sx->lock);

//	if( sx->buffer ) vfree(sx->buffer);
//	sx->buffer=NULL;
	
	up(&sx->lock);
	MOD_DEC_USE_COUNT;
}

static int sxcam_init_done(struct video_device *dev)
{
	return 0;
}

static long sxcam_write(struct video_device *v, const char *buf, unsigned long count, int noblock)
{
	return -EINVAL;
}

static int sxcam_ioctl(struct video_device *dev, unsigned int cmd, void *arg)
{
	struct sxcam_device *sx=(struct sxcam_device *)dev;
	
	switch(cmd)
	{
		case VIDIOCGCAP:
		{
			struct video_capability b;
			strcpy(b.name, "Starlight Xpress");
			b.type = VID_TYPE_CAPTURE|VID_TYPE_MONOCHROME;
			b.channels = 1;
			b.audios = 0;
			b.maxwidth = sx->ccd_width;
			b.maxheight = sx->ccd_height;
			b.minwidth = 10;
			b.minheight = 10;
			if(copy_to_user(arg, &b,sizeof(b)))
				return -EFAULT;
			return 0;
		}
		case VIDIOCGCHAN:
		{
			struct video_channel v;
			if(copy_from_user(&v, arg, sizeof(v)))
				return -EFAULT;
			if(v.channel!=0)
				return -EINVAL;
			v.flags=0;
			v.tuners=0;
			/* Good question.. its composite or SVHS so.. */
			v.type = VIDEO_TYPE_CAMERA;
			strcpy(v.name, "Camera");
			if(copy_to_user(arg, &v, sizeof(v)))
				return -EFAULT;
			return 0;
		}
		case VIDIOCSCHAN:
		{
			int v;
			if(copy_from_user(&v, arg,sizeof(v)))
				return -EFAULT;
			if(v!=0)
				return -EINVAL;
			return 0;
		}
		case VIDIOCGTUNER:
		{
			struct video_tuner v;
			if(copy_from_user(&v, arg, sizeof(v))!=0)
				return -EFAULT;
			if(v.tuner)
				return -EINVAL;
			strcpy(v.name, "Format");
			v.rangelow=0;
			v.rangehigh=0;
			v.flags= 0;
			v.mode = VIDEO_MODE_AUTO;
			if(copy_to_user(arg,&v,sizeof(v))!=0)
				return -EFAULT;
			return 0;
		}
		case VIDIOCSTUNER:
		{
			struct video_tuner v;
			if(copy_from_user(&v, arg, sizeof(v))!=0)
				return -EFAULT;
			if(v.tuner)
				return -EINVAL;
			if(v.mode!=VIDEO_MODE_AUTO)
				return -EINVAL;
			return 0;
		}
		case VIDIOCGPICT:
		{
			struct video_picture p;
			p.colour=0x8000;
			p.hue=0x8000;
			p.brightness=sx->exptime;
			p.depth=sx->depth;
			p.palette=sx->palette;
			if(copy_to_user(arg, &p, sizeof(p)))
				return -EFAULT;
			return 0;
		}
		case VIDIOCSPICT:
		{
			struct video_picture p;
			if(copy_from_user(&p, arg, sizeof(p)))
				return -EFAULT;

			/*
			 *	Sanity check args
			 */
			if ( !((p.depth == 16 && p.palette == VIDEO_PALETTE_GREY) ||
			     (p.depth == 8 && p.palette == VIDEO_PALETTE_GREY) ||
			     (p.depth == 24 && p.palette == VIDEO_PALETTE_RGB24)) )
				return -EINVAL;

			/*
			 *	Now load the camera.
			 */
			sx->exptime = p.brightness;
			sx->depth   = p.depth;
			sx->palette = p.palette;

//			parport_claim_or_block(sx->pdev);
//			down(&sx->lock);			
//			sxcam_setup(sx);
//			up(&sx->lock);
//			parport_release(sx->pdev);
			return 0;
		}
		case VIDIOCSWIN:
		{
			struct video_window vw;
			if(copy_from_user(&vw, arg,sizeof(vw)))
				return -EFAULT;
			if(vw.flags)
				return -EINVAL;
			if(vw.clipcount)
				return -EINVAL;
			if(vw.height<60||vw.height>sx->ccd_height)
				return -EINVAL;
			if(vw.width<80||vw.width>sx->ccd_width)
				return -EINVAL;

			sx->width = vw.width;
			sx->height= vw.height;
			sx->left  = vw.x;
			sx->top   = vw.y;
			
			return 0;

		}
		case VIDIOCGWIN:
		{
			struct video_window vw;
			vw.x=sx->left;
			vw.y=sx->top;
			vw.width =sx->width;
			vw.height=sx->height;
			vw.chromakey=0;
			vw.clipcount=0;
			vw.flags=0;
			if(copy_to_user(arg, &vw, sizeof(vw)))
				return -EFAULT;
			return 0;
		}
		case VIDIOCCAPTURE:
			return -EINVAL;
		case VIDIOCGFBUF:
			return -EINVAL;
		case VIDIOCSFBUF:
			return -EINVAL;
		case VIDIOCKEY:
			return 0;
		case VIDIOCGFREQ:
			return -EINVAL;
		case VIDIOCSFREQ:
			return -EINVAL;
		case VIDIOCGAUDIO:
			return -EINVAL;
		case VIDIOCSAUDIO:
			return -EINVAL;
		default:
			return -ENOIOCTLCMD;
	}
	return 0;
}

static long sxcam_read(struct video_device *dev, char *buf, unsigned long count,  int noblock)
{
	struct sxcam_device *sx=(struct sxcam_device *)dev;
	int len;

	down(&sx->lock);
	parport_claim_or_block(sx->pdev);

	/* Probably should have a semaphore against multiple users */
	len = sxcam_capture(sx, (unsigned char *)buf,count);

	parport_release(sx->pdev);
	up(&sx->lock);

	return len;
}



/* video device template */
static struct video_device sxcam_template=
{
	"Starlight Xpress",
	VID_TYPE_CAPTURE|VID_TYPE_MONOCHROME,
	VID_HARDWARE_QCAM_C,
	sxcam_open,
	sxcam_close,
	sxcam_read,
	sxcam_write,
	NULL,
	sxcam_ioctl,
	NULL,
	sxcam_init_done,
	NULL,
	0,
	0
};

/* Initialize the QuickCam driver control structure. */

static struct sxcam_device *sxcam_init(struct parport *port)
{
	struct sxcam_device *sx;
	
	sx = kmalloc(sizeof(struct sxcam_device), GFP_KERNEL);
	if(sx==NULL)
		return NULL;

	sx->pport = port;
	sx->pdev = parport_register_device(port, "sxcam", NULL, NULL,
					  NULL, 0, NULL);
	if (sx->pdev == NULL) 
	{
		printk(KERN_ERR "sxcam: couldn't register for %s.\n",
		       port->name);
		kfree(sx);
		return NULL;
	}
	
	memcpy(&sx->vdev, &sxcam_template, sizeof(sxcam_template));

	init_MUTEX(&sx->lock);

	sx->ccd_width = SXWIDTH;
	sx->ccd_height = 298;
	sx->width = 500;
	sx->height = 290;
	sx->palette = VIDEO_PALETTE_GREY;
	sx->depth = 16;
	sx->exptime = 1000;
	sx->top = 8;
	sx->left = 38;

	return sx;
}

#define MAX_CAMS 4
static struct sxcam_device *sxcams[MAX_CAMS];
static unsigned int num_cams = 0;


int init_sxcam(struct parport *port)
{
	struct sxcam_device *sx;

	if (num_cams == MAX_CAMS)
	{
		printk(KERN_ERR "Too many Starlight Xpress Cameras (max %d)\n", MAX_CAMS);
		return -ENOSPC;
	}

	sx=sxcam_init(port);
	if(sx==NULL)
		return -ENODEV;
		
	parport_claim_or_block(sx->pdev);

	sxcam_reset(sx);
	
	if(sxcam_detect(sx)==0)
	{
		parport_release(sx->pdev);
		parport_unregister_device(sx->pdev);
		kfree(sx);
		return -ENODEV;
	}
//	sxcam_calibrate(sx);

	parport_release(sx->pdev);
	
	printk(KERN_INFO "Starlight Xpress Camera on %s\n", sx->pport->name);
	
	if(video_register_device(&sx->vdev, VFL_TYPE_GRABBER)==-1)
	{
		parport_unregister_device(sx->pdev);
		kfree(sx);
		return -ENODEV;
	}

	sxcams[num_cams++] = sx;

	return 0;
}

void close_sxcam(struct sxcam_device *sx)
{
	video_unregister_device(&sx->vdev);
	parport_unregister_device(sx->pdev);
	kfree(sx);
}

#ifdef MODULE

static char *parport[MAX_CAMS] = { NULL, };

MODULE_AUTHOR("Holger Jakob <jakob@ph1.uni-koeln.de>");
MODULE_PARM(parport, "1-" __MODULE_STRING(MAX_CAMS) "s");


int init_module(void)
{
	struct parport *port;
	int n;
	if(parport[0] && strncmp(parport[0], "auto", 4)){
		/* user gave parport parameters */
		for(n=0; parport[n] && n<MAX_CAMS; n++){
			char *ep;
			unsigned long r;
			r = simple_strtoul(parport[n], &ep, 0);
			if(ep == parport[n]){
				printk(KERN_ERR
					"sxcam: bad port specifier \"%s\"\n",
					parport[n]);
				continue;
			}
			for (port=parport_enumerate(); port; port=port->next){
				if(r!=port->number)
					continue;
				init_sxcam(port);
				break;
			}
		}
		return (num_cams)?0:-ENODEV;
	} 
	/* no parameter or "auto" */
	for (port = parport_enumerate(); port; port=port->next)
		init_sxcam(port);

	/* Do some sanity checks on the module parameters. */
	if (yieldlines < 1) {
		printk("Starlight Xpress Camera yieldlines was less than 1. Using 1.\n");
		yieldlines = 1;
	}

	return (num_cams)?0:-ENODEV;
}

void cleanup_module(void)
{
	unsigned int i;
	for (i = 0; i < num_cams; i++)
		close_sxcam(sxcams[i]);
}
#else
int __init init_colour_sxcams(struct video_init *unused)
{
	struct parport *port;
        
	for (port = parport_enumerate(); port; port=port->next)
		init_sxcam(port);
	return 0;
}
#endif
